
#' Calculate force of infection
#'
#' This function calculates the force of infection at time t+1, generated by
#' cases with onset at time 1, 2, ..., t, with reproduction numbers R_1, R_2,
#' ..., R_t. Calculations use matrices where independent simulations are stored
#' as separate columns.
#'
#' @param w a `numeric` containing numbers representing the PMF of the serial
#'   interval, starting at day 1, i.e. one day after symptom onset; the vector
#'   should always be at least as long as the largest value of `t` allowed
#'
#' @param cases a `matrix` of `integers` with `t` rows (each row is a date) and
#'   `n_sim` columns (each column is a separate simulation) indicating number of
#'   cases on a given day, in a given simulation
#'
#' @param R a `matrix` of `numeric` with `t` rows (each row is a date) and
#'   `n_sim` columns (each column is a separate simulation) indicating the
#'   reproduction number on a given day, in a given simulation
#'
#' @param t an `integer` indicating the simulation step: incidence will then be 
#'   computed for `t+1` taking into account past cases and R from time point `1`
#'   until `t`
#'
#' @noRd 

compute_force_infection <- function(w, cases, R, t) {
  rev_w <- rev(w)
  ws <- utils::tail(rev_w, t)

  cases <- cases[seq_len(t), , drop = FALSE]
  R <- R[seq_len(t), , drop = FALSE]
  
  lambda <- ws %*% (cases * R)
  as.vector(lambda)
}
