% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/lmer_pi_futvec.R
\name{lmer_pi_futvec}
\alias{lmer_pi_futvec}
\title{Prediction intervals for future observations based on linear random effects models}
\usage{
lmer_pi_futvec(
  model,
  futvec,
  newdat = NULL,
  alternative = "both",
  alpha = 0.05,
  nboot = 10000,
  delta_min = 0.01,
  delta_max = 10,
  tolerance = 0.001,
  traceplot = TRUE,
  n_bisec = 30,
  algorithm = "MS22"
)
}
\arguments{
\item{model}{a random effects model of class lmerMod}

\item{futvec}{an integer vector that defines the structure of the future data based on the
row numbers of the historical data. If \code{length(futvec)} is one, a PI
for one future observation is computed}

\item{newdat}{a \code{data.frame} with the same column names as the historical data
on which  \code{model} depends}

\item{alternative}{either "both", "upper" or "lower". \code{alternative} specifies
if a prediction interval or an upper or a lower prediction limit should be computed}

\item{alpha}{defines the level of confidence (1-\code{alpha})}

\item{nboot}{number of bootstraps}

\item{delta_min}{lower start value for bisection}

\item{delta_max}{upper start value for bisection}

\item{tolerance}{tolerance for the coverage probability in the bisection}

\item{traceplot}{if \code{TRUE}: Plot for visualization of the bisection process}

\item{n_bisec}{maximal number of bisection steps}

\item{algorithm}{either "MS22" or "MS22mod" (see details)}
}
\value{
\code{lmer_pi_futvec()} returns an object of class \code{c("predint", "normalPI")}
with prediction intervals or limits in the first entry (\code{$prediction}).
}
\description{
\code{lmer_pi_futvec()} calculates a bootstrap calibrated prediction interval for one or more
future observation(s) based on linear random effects models. With this approach,
the experimental design of the future data is taken into account (see below).
}
\details{
This function returns bootstrap-calibrated prediction intervals as well as
lower or upper prediction limits.

If \code{algorithm} is set to "MS22", both limits of the prediction interval
are calibrated simultaneously using the algorithm described in Menssen and
Schaarschmidt (2022), section 3.2.4. The calibrated prediction interval is given
as

\deqn{[l,u] = \hat{\mu} \pm q^{calib} \sqrt{\widehat{var}(\hat{\mu}) + \sum_{c=1}^{C+1}
\hat{\sigma}^2_c}}

with \eqn{\hat{\mu}} as the expected future observation (historical mean) and
\eqn{\hat{\sigma}^2_c} as the \eqn{c=1, 2, ..., C} variance components and \eqn{\hat{\sigma}^2_{C+1}}
as the residual variance obtained from the random
effects model fitted with \code{lme4::lmer()} and \eqn{q^{calib}} as the as the bootstrap-calibrated
coefficient used for interval calculation. \cr

If \code{algorithm} is set to "MS22mod", both limits of the prediction interval
are calibrated independently from each other. The resulting prediction interval
is given by

\deqn{[l,u] = \Big[\hat{\mu} - q^{calib}_l \sqrt{\widehat{var}(\hat{\mu}) + \sum_{c=1}^{C+1} \hat{\sigma}^2_c}, \text{   }
\hat{\mu} + q^{calib}_u \sqrt{\widehat{var}(\hat{\mu}) + \sum_{c=1}^{C+1} \hat{\sigma}^2_c} \Big].}

Please note, that this modification does not affect the calibration procedure, if only
prediction limits are of interest. \cr

Be aware that the sampling structure of the historical data must contain the structure of the
future data. This means that the observations per random factor must be less or
equal in the future data compared to the historical data.

This function is an implementation of the PI given in Menssen and Schaarschmidt 2022 section 3.2.4
except that the bootstrap calibration values are drawn from bootstrap samples that
mimic the future data.
}
\examples{

# loading lme4
library(lme4)

# Fitting a random effects model based on c2_dat1
fit <- lmer(y_ijk~(1|a)+(1|b)+(1|a:b), c2_dat1)
summary(fit)

#----------------------------------------------------------------------------

### Prediction interval using c2_dat3 as future data
# without printing c2_dat3 in the output

# Row numbers of the historical data c2_dat1 that define the structure of
# the future data c2_dat3
futvec <- c(1, 2, 4, 5, 10, 11, 13, 14)

# Calculating the PI
\donttest{pred_int <- lmer_pi_futvec(model=fit, futvec=futvec, nboot=100)
summary(pred_int)}

#----------------------------------------------------------------------------

### Calculating the PI with c2_dat3 printed in the output
\donttest{pred_int_new <- lmer_pi_futvec(model=fit, futvec=futvec, newdat=c2_dat3, nboot=100)
summary(pred_int_new)}

#----------------------------------------------------------------------------

### Upper prediction limit for m=1 future observation
\donttest{pred_u <- lmer_pi_futvec(model=fit, futvec=1, alternative="upper", nboot=100)
summary(pred_u)}

#----------------------------------------------------------------------------

# Please note that nboot was set to 100 in order to decrease computing time
# of the example. For a valid analysis set nboot=10000.

}
\references{
Menssen and Schaarschmidt (2022): Prediction intervals for all of M future
observations based on linear random effects models. Statistica Neerlandica,
 \doi{10.1111/stan.12260}
}
