% Generated by roxygen2 (4.1.1): do not edit by hand
% Please edit documentation in R/mean_survival.R
\name{survmean}
\alias{survmean}
\title{Compute mean survival times using extrapolation}
\usage{
survmean(data, surv.breaks = NULL, by.vars = NULL, pophaz = NULL, r = 1,
  agegr.w.breaks = NULL, agegr.w.weights = NULL, ext.breaks = NULL,
  subset = NULL, ...)
}
\arguments{
\item{data}{a data set of splitted records; e.g. output of \code{\link{lexpand}}}

\item{surv.breaks}{passed on to \code{\link{survtab}}; see that help}

\item{by.vars}{a character vector of variables names;
 e.g. \code{by.vars = "sex"} will calculate
mean survival separately for each unique combination of these variables}

\item{pophaz}{a data set of appropriate population hazards as given to
\code{lexpand}; will be used in extrapolation - see Details}

\item{r}{a numeric of length one; multiplies population hazard in \code{pophaz}
by this number; used e.g. \code{r = 1.1} if excess hazard of 10 percent should persist
in extrapolation}

\item{agegr.w.breaks}{a numeric vector of fractional years as \code{[a,b)}
breaks as in \code{survtab}; will be used to determine standardization age group}

\item{agegr.w.weights}{a numeric vector of weights
breaks as in \code{survtab}; will be used to standardize by age group}

\item{ext.breaks}{advanced; a list of breaks (see \code{\link{lexpand}});
used as breaks for extrapolation; see Details}

\item{subset}{a logical condition; e.g. \code{subset = sex == 1};
subsets the data before computations}

\item{...}{any other arguments passed on to \code{survtab} such as
\code{surv.method = "lifetable"} for actuarial estimates of observed survival}
}
\description{
Compute mean survival times using extrapolation
}
\details{
\code{survmean} computes mean survival times.
This is done using a) observed survival estimates computed with \code{survtab}
and b) extrapolated survival probabilities using EdererI method expected
survivals for subjects surviving beyond the roof of \code{surv.breaks},
up to 100 years forward but only up to the 125th birthday by default. The area under
the resulting extrapolated curve is computed via trapezoidal integration,
which is the mean survival time.

For extrapolation, the user must supply a \code{pophaz} data set of population
hazards. The extrapolation itself is essentially done
by splitting the extrapolated observations and merging population hazards
to those records using \code{lexpand}.

The user may compute age-standardized mean survival time estimates using the
\code{agegr.w.breaks} and \code{agegr.w.weights}
parameters, though this is also fairly simple to do by hand via using the
\code{by.vars} argument and merging in the weights yourself.

Note that mean survival is based by default on hazard-based estimates of
observed survival as outlined in \code{survtab}. Unlike with actuarial
estimates, observed survival can never fall to zero using this method.
However, the bias caused by this is likely to be small, and hazard-based
estimation allows for e.g. period method estimates of mean survival time.

\strong{Extrapolation tweaks}

One may tweak the accuracy and length of extrapolation by using \code{ext.breaks}:
By default the survivals of any survivors beyond the roof of \code{surv.breaks}
are extrapolated up to 100 years from the roof of \code{surv.breaks}
or up to their 125th birthday, whichever comes first. The extrapolation
is by default based on the assumption that population hazards supplied
by \code{pophaz} are constant in time periods of length 1/12, 0.25, or 1 years:
if \code{ext.breaks = NULL}, it is internally substituted by

\code{list(fot = c(0:6/12, 0.75, 1:100), age = c(0, 125))}

to be supplied internally to a \code{lexpand} call.

Hence, alternate specifications allow for longer/shorter and more/less
accurate extrapolations. E.g.

\code{ext.breaks = list(fot = seq(0,100,1/12), age = 0:125, per = 1900:2100)}

would ensure a smooth extrapolation and perfect usage of \code{pophaz}.
This will probably not produce results much different from the default, though.
}
\examples{
## take first 5000 subjects in sire data for demonstration
sr <- sire[1:5000, ]
sr$agegr <- cut(sr$dg_age, c(0,45,60,Inf), right=FALSE)
x <- lexpand(sr, breaks=list(fot=seq(0,10,1/12)), pophaz=popmort)
sm <- survmean(x, pophaz=popmort)
## for each level of "agegr" separately:
#sma<- survmean(x, pophaz=popmort, by.vars="agegr")
## automated age-standardised results:
#sms<- survmean(x, pophaz=popmort, agegr.w.breaks=c(0,45,60,Inf))

## visual inspection of how realistic extrapolation is for each stratum;
## grey vertical line points to start of extrapolation;
## solid lines are observed and extrapolated survivals;
## dashed lines are expected survivals
plot(sm)
# plot(sma)
# plot(sms) plots precisely the same as plot(sma)
}
\author{
Joonas Miettinen, Karri Seppa
}

