% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/processWithPraat.R
\name{processWithPraat}
\alias{processWithPraat}
\title{Process a set of intervals with Praat}
\usage{
processWithPraat(
  labbcat.url,
  match.ids,
  start.offsets,
  end.offsets,
  praat.script,
  window.offset,
  gender.attribute = "participant_gender",
  attributes = NULL,
  no.progress = FALSE
)
}
\arguments{
\item{labbcat.url}{URL to the LaBB-CAT instance}

\item{match.ids}{A vector of annotation IDs, e.g. the MatchId column, or the URL column,
of a results set.}

\item{start.offsets}{The start time in seconds, or a vector of start times.}

\item{end.offsets}{The end time in seconds, or a vector of end times.}

\item{praat.script}{Script to run on each match. This may be a single string or a
character vector.}

\item{window.offset}{In many circumstances, you will want some context before and after
the sample start/end time.  For this reason, you can specify a "window offset" -
this is a number of seconds to subtract from the sample start and add to the sample
end time, before extracting that part of the audio for processing. For example, if
the sample starts at 2.0s and ends at 3.0s, and you set the window offset to 0.5s,
then Praat will extract a sample of audio from  1.5s to 3.5s, and do the selected
processing on that sample. The best value for this depends on what the praat.script
is doing; if you are getting formants from  vowels, including some context ensures
that the formants at the edges are more accurate (in LaBB-CAT's web interface, the
default value for this 0.025), but if you're getting max pitch or COG during a
segment, most likely you want a window.offset of 0 to ensure neighbouring segments
don't influence the measurement.}

\item{gender.attribute}{Which participant attribute represents the participant's gender.}

\item{attributes}{Vector of participant attributes to make available to the script. For
example, if you want to use different acoustic parameters depending on what the
gender of the speaker is, including the "participant_gender" attribute will make a
variable called participant_gender$ available to the praat script, whose value will
be the gender of the speaker of that segment.}

\item{no.progress}{TRUE to suppress visual progress bar. Otherwise, progress bar will be
shown when interactive().}
}
\value{
A data frame of acoustic measures, one row for each matchId.
}
\description{
This function instructs the LaBB-CAT server to invoke Praat for a set of sound
intervals, in order to extract acoustic measures.
}
\details{
The exact measurements to return depend on the praat.script that is invoked. This is a
Praat script fragment that will run once for each sound interval specified.

There are functions to allow the generation of a number of pre-defined praat scripts
for common tasks such as formant, pitch, intensity, and centre of gravity -- see
\link{praatScriptFormants}, \link{praatScriptCentreOfGravity}, \link{praatScriptIntensity}
and \link{praatScriptPitch}.

You can provide your own script, either by building a string with your code, or loading
one from a file.

LaBB-CAT prefixes praat.script with code to open a sound file and extract a defined part
of it into a Sound object which is then selected.

LaBB-CAT \verb{Remove's this Sound object after the script finishes executing. Any other objects created by the script must be }Remove'd before the end of the script (otherwise Praat runs out
of memory during very large batches)

LaBB-CAT assumes that all calls to the function 'print' correspond to fields for export
and each field must be printed on its own line. Specifically it scans for lines of the
form:

print 'myOutputVariable' 'newline$'

Variables that can be assumed to be already set in the context of the script are:
\itemize{
\item \emph{windowOffset}
-- the value used for the Window Offset; how much context to include.
\item \emph{windowAbsoluteStart}
-- the start time of the window extracted relative to the start of the original audio file.
\item \emph{windowAbsoluteEnd}
-- the end time of the window extracted relative to the start of the original audio file.
\item \emph{windowDuration}
-- the duration of the window extracted (including window offset).
\item \emph{targetAbsoluteStart}
-- the start time of the target interval relative to the start of the original audio file.
\item \emph{targetAbsoluteEnd}
-- the end time of the target interval relative to the start of the original audio file.
\item \emph{targetStart}
-- the start time of the target interval relative to the start of the window extracted.
\item \emph{targetEnd}
-- the end time of the target interval relative to the start of the window extracted.
\item \emph{targetDuration}
-- the duration of the target interval.
\item \emph{sampleNumber}
-- the number of the sample within the set of samples being processed.
\item \emph{sampleName$}
-- the name of the extracted/selected Sound object.
}
}
\examples{
\dontrun{
## Perform a search
results <- getMatches(labbcat.url, list(segment="I"))

## get F1 and F2 for the mid point of the vowel
formants <- processWithPraat(
       labbcat.url,
       results$MatchId, results$Target.segment.start, results$Target.segment.end,
       praatScriptFormants())

## get first 3 formants at three points during the sample, the mean, min, and max
## pitch, the max intensity, and the CoG using powers 1 and 2 
acoustic.measurements <- processWithPraat(
       labbcat.url,
       results$MatchId, results$Target.segment.start, results$Target.segment.end,
       paste(
           praatScriptFormants(c(1,2,3), c(0.25,0.5,0.75)),
           praatScriptPitch(get.mean=TRUE, get.minimum=TRUE, get.maximum=TRUE),
           praatScriptIntensity(),
           praatScriptCentreOfGravity(powers=c(1.0,2.0))),
       window.offset=0.5)

## execute a custom script loaded form a file
acoustic.measurements <- processWithPraat(
       labbcat.url,
       results$MatchId, results$Target.segment.start, result$Target.segment.end,
       readLines("acousticMeasurements.praat"))
}
}
\seealso{
Other Praat-related functions: 
\code{\link{praatScriptCentreOfGravity}()},
\code{\link{praatScriptFastTrack}()},
\code{\link{praatScriptFormants}()},
\code{\link{praatScriptIntensity}()},
\code{\link{praatScriptPitch}()}
}
\concept{Praat-related functions}
\keyword{praat}
