% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/nowcast.R
\name{nowcast}
\alias{nowcast}
\title{Nowcasting of a quarterly time series using a dynamic factor model.}
\usage{
nowcast(y, x, q = NULL, r = NULL, p = NULL, method = "2s",
  blocks = NULL, frequency = NULL, oldFactorsParam = NULL,
  oldRegParam = NULL)
}
\arguments{
\item{y}{Stationary quarterly time series.}

\item{x}{A monthly time series matrix (\code{mts}) representing regressors variables. The series must be stationary.}

\item{q}{Dynamic rank. Number of error terms.}

\item{r}{number of commom factors.}

\item{p}{AR order of factor model.}

\item{method}{There are three options: \code{"2s"} (two stages without factors aggregation as in Giannone et al. 2008); \code{"2s_agg"} (two stages with factors aggregation); \code{"EM"} (Expected Maximization as in Bańbura et al. 2011).}

\item{blocks}{a binary matrix Nx3 that characterizes the regressors variables in global (1st column), nominal (2nd column) and real (3rd column). If \code{NULL}, the matrix assume 1 for all cells.}

\item{frequency}{A vector of integers indicating the frequency of the variables: 4 for quarterly, 12 for monthly.}

\item{oldFactorsParam}{a list containing estimated factors parameters from nowcast function.}

\item{oldRegParam}{a list containing estimated regression parameters from nowcast function.}
}
\value{
A \code{list} containing two elements:
\item{yfcst}{the original \code{y} series and its in-sample and out-of-sample estimations.}
\item{reg}{regression model between \code{y} and the estimated factors. Not available for EM method.}
\item{factors}{the estimated factors and DFM model coefficients.}
\item{xfcst}{the original regressors and their out-of-sample estimations.}
\item{month_y}{the monthly measure for quarterly \code{y} variable. Only available for EM method.}
}
\description{
Estimate nowcasting and forecasting for a quarterly series. For more details read the Vignettes.
}
\examples{
\dontrun{
### Method 2s (without aggregation)
gdp <- month2qtr(x = USGDP$base[,"RGDPGR"])
gdp_position <- which(colnames(USGDP$base) == "RGDPGR")
base <- Bpanel(base = USGDP$base[,-gdp_position],
               trans = USGDP$legend$Transformation[-gdp_position],
               aggregate = TRUE)
now2s <- nowcast(y = gdp, x = base, r = 2, p = 2, q = 2, method = '2s')

### Method 2s_agg (with aggregation)
base <- Bpanel(base = USGDP$base[,-gdp_position],
               trans = USGDP$legend$Transformation[-gdp_position],
               aggregate = FALSE)
now2s_agg <- nowcast(y = gdp, x = base, r = 2, p = 2, q = 2, method = '2s_agg')

### Method EM
# Replication of the NY FED nowcast
data(NYFED)

base <- NYFED$base
blocks <- NYFED$blocks$blocks
trans <- NYFED$legend$Transformation
frequency <- NYFED$legend$Frequency

x <- Bpanel(base = base, trans = trans, NA.replace = F, na.prop = 1)

nowEM <- nowcast(y = "GDPC1", x = x, r = 1, p = 1, method = "EM",
 blocks = blocks, frequency = frequency)

}
}
\references{
Giannone, D., Reichlin, L., & Small, D. (2008). Nowcasting: The real-time informational content of macroeconomic data. Journal of Monetary Economics, 55(4), 665-676.<doi:10.1016/j.jmoneco.2008.05.010>

Bańbura, M., & Rünstler, G. (2011). A look into the factor model black box: publication lags and the role of hard and soft data in forecasting GDP. International Journal of Forecasting, 27(2), 333-346. <doi:10.1016/j.ijforecast.2010.01.011>

Bańbura M., Giannone, D. & Reichlin, L. (2011). Nowcasting, in Michael P. Clements and David F. Hendry, editors, Oxford Handbook on Economic Forecasting, pages 193-224, January 2011. <doi:10.1093/oxfordhb/9780195398649.001.0001>
}
\seealso{
\code{\link[nowcasting]{base_extraction}}
}
