% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/update_hetGP.R
\name{update.hetGP}
\alias{update.hetGP}
\title{Update \code{"hetGP"}-class model fit with new observations}
\usage{
\method{update}{hetGP}(object, Xnew, Znew, ginit = 0.01, lower = NULL,
  upper = NULL, noiseControl = NULL, settings = NULL, known = NULL,
  maxit = 100, method = "quick", ...)
}
\arguments{
\item{object}{previously fit \code{"hetGP"}-class model}

\item{Xnew}{matrix of new design locations; \code{ncol(Xnew)} must match the input dimension encoded in \code{object}}

\item{Znew}{vector new observations at those design locations, of length \code{nrow(X)}. \code{NA}s can be passed, see Details}

\item{ginit}{minimal value of the smoothing parameter (i.e., nugget of the noise process) for optimization initialisation.
It is compared to the \code{g} hyperparameter in the object.}

\item{lower, upper, noiseControl, settings, known}{optional bounds for mle optimization, see \code{\link[hetGP]{mleHetGP}}. 
If not provided, they are extracted from the existing model}

\item{maxit}{maximum number of iterations for the internal L-BFGS-B optimization method; see \code{\link{optim}} for more details}

\item{method}{one of \code{"quick"}, \code{"mixed"} see Details.}

\item{...}{no other argument for this method.}
}
\description{
Fast update of existing \code{hetGP} model with new observations.
}
\details{
The update can be performed with or without re-estimating hyperparameter.
In the first case, \code{\link[hetGP]{mleHetGP}} is called, based on previous values for initialization. 
The only missing values are the latent variables at the new points, that are initialized based on two possible update schemes in \code{method}:
\itemize{
  \item \code{"quick"} the new delta value is the predicted nugs value from the previous noise model;
  \item \code{"mixed"} new values are taken as the barycenter between prediction given by the noise process and empirical variance. 
}
The subsequent number of MLE computations can be controlled with \code{maxit}.

In case hyperparameters need not be updated, \code{maxit} can be set to \code{0}. 
In this case it is possible to pass \code{NA}s in \code{Znew}, then the model can still be used to provide updated variance predictions.
}
\examples{
##------------------------------------------------------------
## Sequential update example
##------------------------------------------------------------
set.seed(42)

## Spatially varying noise function
noisefun <- function(x, coef = 1){
  return(coef * (0.05 + sqrt(abs(x)*20/(2*pi))/10))
}

## Initial data set
nvar <- 1
n <- 20
X <- matrix(seq(0, 2 * pi, length=n), ncol = 1)
mult <- sample(1:10, n, replace = TRUE)
X <- rep(X, mult)
Z <- sin(X) + rnorm(length(X), sd = noisefun(X))

## Initial fit
testpoints <- matrix(seq(0, 2*pi, length = 10*n), ncol = 1)
model <- model_init <- mleHetGP(X = X, Z = Z, lower = rep(0.1, nvar), 
  upper = rep(50, nvar), maxit = 1000, settings = list(hardpenalty = TRUE))

## Visualizing initial predictive surface
preds <- predict(x = testpoints, model_init) 
plot(X, Z)
lines(testpoints, preds$mean, col = "red")

## 10 fast update steps
nsteps <- 5
npersteps <- 10
for(i in 1:nsteps){
  newIds <- sort(sample(1:(10*n), npersteps))
  
  newX <- testpoints[newIds, drop = FALSE] 
  newZ <- sin(newX) + rnorm(length(newX), sd = noisefun(newX))
  points(newX, newZ, col = "blue", pch = 20)
  model <- update(object = model, Xnew = newX, Znew = newZ)
  X <- c(X, newX)
  Z <- c(Z, newZ)
  plot(X, Z)
  print(model$nit_opt)
}

## Final predictions after 10 updates
preds_fin <- predict(x=testpoints, model) 

## Visualizing the result by augmenting earlier plot
lines(testpoints, preds_fin$mean, col = "blue")
lines(testpoints, qnorm(0.05, preds_fin$mean, sqrt(preds_fin$sd2)), col = "blue", lty = 2)
lines(testpoints, qnorm(0.95, preds_fin$mean, sqrt(preds_fin$sd2)), col = "blue", lty = 2)
lines(testpoints, qnorm(0.05, preds_fin$mean, sqrt(preds_fin$sd2 + preds_fin$nugs)), 
  col = "blue", lty = 3)
lines(testpoints, qnorm(0.95, preds_fin$mean, sqrt(preds_fin$sd2 + preds_fin$nugs)), 
  col = "blue", lty = 3)

## Now compare to what you would get if you did a full batch fit instead
model_direct <-  mleHetGP(X = X, Z = Z, maxit = 1000, settings = list(hardpenalty = TRUE),
                          lower = rep(0.1, nvar), upper = rep(50, nvar),
                          init = list(theta = model_init$theta, k_theta_g = model_init$k_theta_g))
preds_direct <- predict(x = testpoints, model_direct)
print(model_direct$nit_opt)
lines(testpoints, preds_direct$mean, col = "green")
lines(testpoints, qnorm(0.05, preds_direct$mean, sqrt(preds_direct$sd2)), col = "green", 
  lty = 2)
lines(testpoints, qnorm(0.95, preds_direct$mean, sqrt(preds_direct$sd2)), col = "green", 
  lty = 2)
lines(testpoints, qnorm(0.05, preds_direct$mean, sqrt(preds_direct$sd2 + preds_direct$nugs)), 
  col = "green", lty = 3)
lines(testpoints, qnorm(0.95, preds_direct$mean, sqrt(preds_direct$sd2 + preds_direct$nugs)), 
  col = "green", lty = 3)
lines(testpoints, sin(testpoints), col = "red", lty = 2)

## Compare outputs
summary(model_init)
summary(model)
summary(model_direct)

}
