% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/confidence.MRFA.R
\name{confidence.MRFA}
\alias{confidence.MRFA}
\title{Confidence Interval for Multiresolution Functional ANOVA (MRFA) Model}
\usage{
confidence.MRFA(object, xnew, X, lambda = object$lambda,
  conf.level = 0.95, var.estimation = c("rss", "cv", "posthoc")[1],
  w.estimation = c("cv", "nugget")[1], K = 5, nugget = 1e-06,
  parallel = FALSE, verbose = FALSE)
}
\arguments{
\item{object}{a class MRFA object estimated by \code{MRFA_fit}.}

\item{xnew}{a testing matrix with dimension \code{n_new} by \code{d} in which each row corresponds to a predictive location.}

\item{X}{input for \code{MRFA_fit}.}

\item{lambda}{a value. The default is \code{min(object$lambda)}.}

\item{conf.level}{a value specifying confidence level of the confidence interval. The default is 0.95.}

\item{var.estimation}{a character string specifying the estimation method for variance. "rss" specifies residual sum of squares, "cv" specifies a cross-validation method with \code{K} fold, and "posthoc" specifies a post-hoc estimation method. The default is "rss".}

\item{w.estimation}{a character string specifying the estimation method for weights w. "cv" specifies a cross-validation method with \code{K} fold, and "nugget" specifies a least square error method with nugget=\code{nugget}. The default is "cv".}

\item{K}{a positive integer specifying the number of folds.}

\item{nugget}{a value specifying the nugget value for \code{w.estimation}. The default is 1e-6. It only works when \code{w.estimation="nugget"}.}

\item{parallel}{logical. If \code{TRUE}, apply function in parallel using parallel backend provided by foreach.}

\item{verbose}{logical. If \code{TRUE}, additional diagnostics are printed.}
}
\value{
\item{lower bound}{a vector with length \code{n_new} displaying lower bound of predicted responses at locations \code{xnew}.}
\item{upper bound}{a vector with length \code{n_new} displaying upper bound of predicted responses at locations \code{xnew}.}
\item{conf.level}{as above.}
}
\description{
The function computes the confidence intervals of predicted responses (only works for linear regression model).
}
\details{
When The details about \code{var.estimation} and \code{w.estimation} can be seen in Sung et al. (2017+).
}
\examples{
\dontrun{

#####             Testing function: OTL circuit function                      #####
#####   Thanks to Sonja Surjanovic and Derek Bingham, Simon Fraser University #####
otlcircuit <- function(xx)
{
  Rb1  <- 50   + xx[1] * 100
  Rb2  <- 25   + xx[2] * 45
  Rf   <- 0.5  + xx[3] * 2.5
  Rc1  <- 1.2  + xx[4] * 1.3
  Rc2  <- 0.25 + xx[5] * 0.95
  beta <- 50   + xx[6] * 250

  Vb1 <- 12*Rb2 / (Rb1+Rb2)
  term1a <- (Vb1+0.74) * beta * (Rc2+9)
  term1b <- beta*(Rc2+9) + Rf
  term1 <- term1a / term1b

  term2a <- 11.35 * Rf
  term2b <- beta*(Rc2+9) + Rf
  term2 <- term2a / term2b

  term3a <- 0.74 * Rf * beta * (Rc2+9)
  term3b <- (beta*(Rc2+9)+Rf) * Rc1
  term3 <- term3a / term3b

  Vm <- term1 + term2 + term3
  return(Vm)
}



library(MRFA)
#####   training data and testing data   #############
set.seed(2)
n <- 100; n_new <- 10; d <- 6
X.train <- matrix(runif(d*n), ncol = d)
Y.train <- apply(X.train, 1, otlcircuit)
X.test <- matrix(runif(d*n_new), ncol = d)
Y.test <- apply(X.test, 1, otlcircuit)

#####   Fitting    #####
MRFA_model <- MRFA_fit(X.train, Y.train)

#####   Prediction   ######
Y.pred <- predict(MRFA_model, X.test, lambda = min(MRFA_model$lambda))$y_hat
print(sqrt(mean((Y.test - Y.pred)^2)))

### confidence interval ###
conf.interval <- confidence.MRFA(MRFA_model, X.test, X.train, lambda = min(MRFA_model$lambda))
print(conf.interval)
}
}
\seealso{
\code{\link{MRFA_fit}} for fitting of a multi-resolution functional ANOVA model; \code{\link{predict.MRFA}} for prediction of a multi-resolution functional ANOVA model.
}
\author{
Chih-Li Sung <iamdfchile@gmail.com>
}
