% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/generic.R
\name{mams.sim}
\alias{mams.sim}
\title{Simulating multi-arm multi-stage designs}
\usage{
mams.sim(obj=NULL,nsim=NULL, nMat=NULL,
u=NULL, l=NULL, pv=NULL, deltav=NULL, sd=NULL, ptest=NULL,
parallel=NULL, H0=NULL, K = NULL)
}
\arguments{
\item{obj}{an object of class \code{MAMS}. The parameters/design of the
considered in the output of the \code{mams()} function are considered as
reference for the simulation. If other parameters are given, their values
\emph{override} the parameters of the \code{MAMS} object}

\item{nsim}{Number of simulations (default=`50000``).}

\item{nMat}{Jx(K+1) dimensional matrix of observed/expected sample sizes.
Rows correspond to stages and columns to arms. First column is control
(default: \code{NULL}).}

\item{u}{Vector of previously used upper boundaries (default=\code{NULL}).}

\item{l}{Vector of previously used upper boundaries (default=\code{NULL}).}

\item{pv}{Vector of size K of true treatment effects on the probability
scale. See Details (default=\code{NULL}).}

\item{deltav}{Vector of size K of true treatment effects on the traditional
scale. See Details (default=\code{NULL}).}

\item{sd}{Standard deviation. See Details (default=\code{NULL}).}

\item{ptest}{Vector of treatment numbers for determining power.
For example, c(1, 2) will count rejections of one or both hypotheses for
testing treatments 1 and 2 against control (default=\code{1}).}

\item{parallel}{if \code{TRUE} (default), allows parallelization of the
computation via a user-defined strategy specified by means of the function
\code{\link[future]{plan}}. If not set differently, the default
strategy is \code{sequential}, which corresponds to a computation without
parallelization.}

\item{H0}{if \code{TRUE} (default), the simulation also considers the case with
all effect sizes set to 0.}

\item{K}{Allocation for treatment arms (used only with method = "dtl")}
}
\value{
An object containing the following components:
\item{l}{Lower boundary.}
\item{u}{Upper boundary.}
\item{n}{Sample size on control in stage 1.}
\item{N}{Maximum total sample size.}
\item{K}{Number of experimental treatments.}
\item{J}{Number of stages in the trial.}
\item{rMat}{Matrix of allocation ratios. First row corresponds to control
and second row to experimental treatments.}
\item{nsim}{Number of simulation runs.}
\item{typeI}{The proportion any hypothesis is rejected.}
\item{power}{The proportion the first hypothesis is rejected and the
corresponding test statistic is largest.}
\item{ptest}{The vector \code{ptest}.}
\item{prop.rej}{The proportion of times at least one of the hypothesis
specified by \code{ptest} is rejected.}
\item{exss}{The expected sample size.}
}
\description{
The function simulates multi-arm multi-stage designs and estimates power and
expected sample size.
}
\details{
This function simulates multi-arm multi-stage studies for a given matrix of
sample sizes and boundaries given by the vectors \code{u} and  \code{l}.
The effect difference between each experimental treatment and control is
given by \code{pv} and is parameterized as
\eqn{P(X_k > X_0 ) = p}{P(X_k > X_0 ) = p}.
That is the probability of a randomly selected person on treatment k
observing a better outcome than a random person on control.
For \code{pv=rep(0.5,4)} the experimental treatments and control perform equally
well (i.e. the global null hypothesis is true).
The advantage of this parameterization is that no knowledge about the
variance is required. To convert traditional effect sizes,
\eqn{\delta}{delta} to this format use
\eqn{p=\Phi(\frac{\delta}{\sqrt{2}\sigma})}{Phi(delta/(2^0.5*sigma))}.
Alternatively, the effect size can also be specified directly on the
traditional scale of \code{deltav} with an additional specification of
the standard deviation \code{sd}.

he function returns the probability of rejecting any hypothesis (\code{typeI}),
the power to reject the first hypothesis when the first treatment has the
largest estimated effect, the proportion of rejections of the hypothesis
specified by \code{ptest} (\code{prop.rej}) as well as the expected
sample size.
}
\examples{
\donttest{
# Note that some of these examples may take a few minutes to run

# 2-stage design with O'Brien & Fleming efficacy and zero futility boundary
# with equal sample size per arm and stage. Design can be found using
# mams(K=4, J=2, alpha=0.05, power=0.9, r=1:2, r0=1:2, ushape="obf", 
     # lshape="fixed",
     # lfix=0, p=0.65, p0=0.55)

# under global null hypothesis (using the pv scale)
mams.sim(nsim=10000, nMat=matrix(c(44, 88), nrow=2, ncol=5), 
             u=c(3.068, 2.169),
             l=c(0.000, 2.169), pv=rep(0.5, 4), ptest=1)

# under global null hypothesis (using the deltav scale)
mams.sim(nsim=10000, nMat=matrix(c(44, 88), nrow=2, ncol=5), 
        u=c(3.068, 2.169),
        l=c(0.000, 2.169), pv=NULL, deltav=rep(0, 4), sd=1, ptest=1)

# under LFC
mams.sim(nsim=10000, nMat=matrix(c(44, 88), nrow=2, ncol=5), 
        u=c(3.068, 2.169),
        l=c(0.000, 2.169), pv=c(0.65, 0.55, 0.55, 0.55), ptest=1:2)

# when all treatments doing similarly well
mams.sim(nsim=10000, nMat=matrix(c(44, 88), nrow=2, ncol=5),
        u=c(3.068, 2.169),
        l=c(0.000, 2.169), pv=c(0.63, 0.62, 0.60, 0.61), ptest=4)

##
## example considering different parallelisation strategies
##

# parallel = FALSE (future framework not used)
set.seed(1)
system.time(
 print(mams.sim(nsim=25000, nMat=matrix(c(44, 88), nrow=2, ncol=5), 
                u=c(3.068, 2.169),
                l=c(0.000, 2.169), pv=c(0.65, 0.55, 0.55, 0.55),
                ptest=1:2, parallel=FALSE))
)
# parallel = TRUE (default) with default strategy (sequential computation)
plan(sequential)
set.seed(1)
system.time(
 print(mams.sim(nsim=25000, nMat=matrix(c(44, 88), nrow=2, ncol=5), 
                u=c(3.068, 2.169),
                l=c(0.000, 2.169), pv=c(0.65, 0.55, 0.55, 0.55), ptest=1:2))
)
# parallel = TRUE (default) with multisession strategy (parallel computation)
plan(multisession)
set.seed(1)
system.time(
 print(mams.sim(nsim=25000, nMat=matrix(c(44, 88), nrow=2, ncol=5),
                u=c(3.068, 2.169),
                l=c(0.000, 2.169), pv=c(0.65, 0.55, 0.55, 0.55), ptest=1:2))
)
plan("default")
}
}
\references{
Jaki T., Pallmann P., and Magirr D. (2019), \emph{The R Package MAMS for
Designing Multi-Arm Multi-Stage Clinical Trials}, \bold{Journal of
Statistical Software}, 88(4), 1-25. Link:
\doi{10.18637/jss.v088.i04}

Magirr D., Jaki T., and Whitehead J. (2012), \emph{A generalized Dunnett test
for multi-arm multi-stage clinical studies with treatment selection},
\bold{Biometrika}, 99(2), 494-501. Link:
\doi{10.1093/biomet/ass002}
}
\seealso{
\link{mams}, \link[=mams]{MAMS}.
}
\author{
Thomas Jaki, Dominic Magirr and Dominique-Laurent Couturier
}
\keyword{design}
