\name{dbVariance}
\alias{dbVariance}
\title{Covariance matrix of cell counts in DNA database comparison}
\description{
  Computes the covariance matrix for the cell counts when comparing DNA
  profiles in a DNA database. For every pair of DNA profiles in a
  database the number of matching and partial matching loci is
  recorded. A match is declared if the two DNA profiles coincide for
  both alleles in a locus and a partial-match is recorded if only one
  allele is shared between the profiles. With a total of L loci the
  number of matching loci is 0,...,L and partial number of matches is
  0,...,L-m, where m is the number of matching loci. The expression is
  given by:
  \deqn{latex}{%
    Var(M) = {n\choose2}Var[M(G_{i_1},G_{i_2})] +
    6*{n\choose3}Cov[M(G_{i_1},G_{i_2}),M(G_{i_1},G_{i_3})] +
    6*{n\choose4}Cov[M(G_{i_1},G_{i_2}),M(G_{i_3},G_{i_4})]
  } 
}
\usage{
dbVariance(probs,theta=0.03,n=1,collapse=FALSE)
}
\arguments{
  \item{probs}{List of vectors with allele probabilities for each locus}
  \item{theta}{The coancestery coefficient. If a vector of different
    theta values are supplied a list of covariance matrices is
    returned. Note it is faster to give a vector of theta values as
    argument than calculating each matrix at the time.}
  \item{n}{Number of DNA profiles in the database. If n=1 is supplied a
    list of the components for computing the variance is returned. That
    is, the variance and two covariances on the right hand side of the
    equation above.}
  \item{collapse}{Logical, default FALSE. If TRUE the covariance matrix
    is collapsed such that it relates to (2*m+p)-vectors of total number
    of matching alleles rather than (m,p)-matrix.}
}
\details{
  Computes the covariance matrix of the cell counts using a recursion
  formula. See Tvedebrink et al (2011) for details.
}
\references{
  T Tvedebrink, PS Eriksen, J Curran, HS Mogensen, N Morling. 'Analysis
  of matches and partial-matches in Danish DNA reference profile
  database'. Forensic Science International: Genetics, 2011.
}
\value{
  Returns a covariance matrix for the cell counts.
}

\author{James Curran and Torben Tvedebrink}

\examples{
  \dontrun{
  ## Simulate some allele frequencies:
  freqs <-  replicate(10, { g = rgamma(n=10,scale=4,shape=3); g/sum(g)}, simplify=FALSE)
  ## List of elements needed to compute the covariance matrix.
  ## Useful option when the covariance needs to be computed for varying
  ## database sizes but for identical theta-value.
  comps <- dbVariance(freqs,theta=0.03,n=1)
  ## Covariance for a DB with 1000 DNA profiles
  cov1000 <- dbVariance(freqs,theta=0.03,n=1000)
  ## The result is the same as:
  comps1000 <- choose(1000,2)*comps$V1 + 6*choose(1000,3)*comps$V2 + 6*choose(1000,4)*comps$V3
  }
}

