% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ddfMLR.R
\name{ddfMLR}
\alias{ddfMLR}
\title{DDF detection for nominal data.}
\usage{
ddfMLR(Data, group, focal.name, key, type = "both", match = "zscore", anchor = NULL,
       purify = FALSE, nrIter = 10, p.adjust.method = "none",
       alpha = 0.05, parametrization)
}
\arguments{
\item{Data}{data.frame or matrix: dataset which rows represent
unscored examinee answers (nominal) and columns correspond to the
items. In addition, \code{Data} can hold the vector of group
membership.}

\item{group}{numeric or character: a dichotomous vector of the same
length as \code{nrow(Data)} or a column identifier of
\code{Data}.}

\item{focal.name}{numeric or character: indicates the level of
\code{group} which corresponds to focal group.}

\item{key}{character: the answer key. Each element corresponds to
the correct answer of one item.}

\item{type}{character: type of DDF to be tested. Either
\code{"both"} for uniform and non-uniform DDF (i.e., difference
in parameters \code{"a"} and \code{"b"}) (default), or
\code{"udif"} for uniform DDF only (i.e., difference in
difficulty parameter \code{"b"}), or \code{"nudif"} for
non-uniform DDF only (i.e., difference in discrimination
parameter \code{"a"}). Can be specified as a single value (for
all items) or as an item-specific vector.}

\item{match}{numeric or character: matching criterion to be used as
an estimate of trait. Can be either \code{"zscore"} (default,
standardized total score), \code{"score"} (total test score), or
vector of the same length as number of observations in
\code{Data}.}

\item{anchor}{numeric or character: specification of DDF free
items. Either \code{NULL} (default), or a vector of item names
(column names of \code{Data}), or item identifiers (integers
specifying the column number) determining which items are
currently considered as anchor (DDF free) items. Argument is
ignored if \code{match} is not \code{"zscore"} or \code{"score"}.}

\item{purify}{logical: should the item purification be applied?
(default is \code{FALSE}).}

\item{nrIter}{numeric: the maximal number of iterations in the item
purification (default is 10).}

\item{p.adjust.method}{character: method for multiple comparison
correction. Possible values are \code{"holm"}, \code{"hochberg"},
\code{"hommel"}, \code{"bonferroni"}, \code{"BH"}, \code{"BY"},
\code{"fdr"}, and \code{"none"} (default). For more details see
\code{\link[stats]{p.adjust}}.}

\item{alpha}{numeric: significance level (default is 0.05).}

\item{parametrization}{deprecated. Use
\code{\link[difNLR]{coef.ddfMLR}} for different
parameterizations.}
}
\value{
The \code{ddfMLR()} function returns an object of class
  \code{"ddfMLR"}. The output including values of the test
  statistics, p-values, and items marked as DDF is displayed by the
  \code{print()} method.

A list of class \code{"ddfMLR"} with the following arguments:
\describe{
  \item{\code{Sval}}{the values of likelihood ratio test statistics.}
  \item{\code{mlrPAR}}{the estimates of final model.}
  \item{\code{mlrSE}}{standard errors of the estimates of final model.}
  \item{\code{parM0}}{the estimates of null model.}
  \item{\code{parM1}}{the estimates of alternative model.}
  \item{\code{llM0}}{log-likelihood of null model.}
  \item{\code{llM1}}{log-likelihood of alternative model.}
  \item{\code{AIC0}}{AIC of null model.}
  \item{\code{AIC1}}{AIC of alternative model.}
  \item{\code{BIC0}}{BIC of null model.}
  \item{\code{BIC1}}{BIC of alternative model.}
  \item{\code{DDFitems}}{either the column identifiers of the items which were detected as DDF, or
  \code{"No DDF item detected"} in case no item was detected as DDF.}
  \item{\code{type}}{character: type of DDF that was tested.}
  \item{\code{anchor}}{DIF free items specified by the \code{anchor} and \code{purify}.}
  \item{\code{purification}}{\code{purify} value.}
  \item{\code{nrPur}}{number of iterations in item purification process. Returned only if \code{purify}
  is \code{TRUE}.}
  \item{\code{ddfPur}}{a binary matrix with one row per iteration of item purification and one column per item.
  \code{"1"} in i-th row and j-th column means that j-th item was identified as DDF in i-th iteration. Returned only
  if \code{purify} is \code{TRUE}.}
  \item{\code{conv.puri}}{logical indicating whether item purification process converged before the maximal number
  \code{nrIter} of iterations. Returned only if \code{purify} is \code{TRUE}.}
  \item{\code{p.adjust.method}}{character: method for multiple comparison correction which was applied.}
  \item{\code{pval}}{the p-values by likelihood ratio test.}
  \item{\code{adj.pval}}{the adjusted p-values by likelihood ratio test using \code{p.adjust.method}.}
  \item{\code{df}}{the degress of freedom of likelihood ratio test.}
  \item{\code{alpha}}{numeric: significance level.}
  \item{\code{Data}}{the data matrix.}
  \item{\code{group}}{the vector of group membership.}
  \item{\code{group.names}}{levels of grouping variable.}
  \item{\code{key}}{key of correct answers.}
  \item{\code{match}}{matching criterion.}
  }

For an object of class \code{"ddfMLR"} several methods are available (e.g. \code{methods(class = "ddfMLR")}).
}
\description{
Performs DDF detection procedure for nominal data
  based on multinomial log-linear regression model and likelihood
  ratio test of a submodel.
}
\details{
Performs DDF detection procedure for nominal data based on
multinomial log-linear regression model and likelihood ratio test
of submodel. Probability of selection the \eqn{k}-th category
(distractor) is
\deqn{P(y = k) = exp((a_k + a_kDif * g) * (x - b_k - b_kDif * g))) / (1 + \sum exp((a_l + a_lDif * g) * (x - b_l - b_lDif * g))), }
where \eqn{x} is by default standardized total score (also called
Z-score) and \eqn{g} is a group membership. Parameters \eqn{a_k}
and \eqn{b_k} are discrimination and difficulty for the \eqn{k}-th
category. Terms \eqn{a_kDif} and \eqn{b_kDif} then represent
differences between two groups (reference and focal) in relevant
parameters. Probability of correct answer (specified in argument
\code{key}) is
\deqn{P(y = k) = 1/(1 + \sum exp((a_l + a_lDif * g)*(x - b_l - b_lDif * g))). }
Parameters are estimated via neural networks. For more details see
\code{\link[nnet]{multinom}}.

Missing values are allowed but discarded for item estimation. They
must be coded as \code{NA} for both, \code{Data} and \code{group}
arguments.
}
\examples{
\dontrun{
# loading data
data(GMATtest, GMATkey)
Data <- GMATtest[, 1:20] # items
group <- GMATtest[, "group"] # group membership variable
key <- GMATkey # correct answers

# testing both DDF effects
(x <- ddfMLR(Data, group, focal.name = 1, key))

# graphical devices
plot(x, item = "Item1", group.names = c("Group 1", "Group 2"))
plot(x, item = x$DDFitems)
plot(x, item = 1)

# estimated parameters
coef(x)
coef(x, SE = TRUE)
coef(x, SE = TRUE, simplify = TRUE)

# AIC, BIC, log-likelihood
AIC(x)
BIC(x)
logLik(x)

# AIC, BIC, log-likelihood for the first item
AIC(x, item = 1)
BIC(x, item = 1)
logLik(x, item = 1)

# testing both DDF effects with Benjamini-Hochberg adjustment method
ddfMLR(Data, group, focal.name = 1, key, p.adjust.method = "BH")

# testing both DDF effects with item purification
ddfMLR(Data, group, focal.name = 1, key, purify = TRUE)

# testing uniform DDF effects
ddfMLR(Data, group, focal.name = 1, key, type = "udif")
# testing non-uniform DDF effects
ddfMLR(Data, group, focal.name = 1, key, type = "nudif")

# testing both DDF effects with total score as matching criterion
ddfMLR(Data, group, focal.name = 1, key, match = "score")
}

}
\references{
Agresti, A. (2010). Analysis of ordinal categorical data. Second edition. John Wiley & Sons.

Hladka, A. (2021). Statistical models for detection of differential item functioning. Dissertation thesis.
Faculty of Mathematics and Physics, Charles University.

Hladka, A. & Martinkova, P. (2020). difNLR: Generalized logistic regression models for DIF and DDF detection.
The R Journal, 12(1), 300--323, \doi{10.32614/RJ-2020-014}.
}
\seealso{
\code{\link[difNLR]{plot.ddfMLR}} for graphical representation of item characteristic curves. \cr
\code{\link[difNLR]{coef.ddfMLR}} for extraction of item parameters with their standard errors. \cr
\code{\link[difNLR]{logLik.ddfMLR}}, \code{\link[difNLR]{AIC.ddfMLR}}, \code{\link[difNLR]{BIC.ddfMLR}}
for extraction of log-likelihood and information criteria. \cr

\code{\link[stats]{p.adjust}} for multiple comparison corrections. \cr
\code{\link[nnet]{multinom}} for estimation function using neural networks.
}
\author{
Adela Hladka (nee Drabinova) \cr
Institute of Computer Science of the Czech Academy of Sciences \cr
Faculty of Mathematics and Physics, Charles University \cr
\email{hladka@cs.cas.cz} \cr

Patricia Martinkova \cr
Institute of Computer Science of the Czech Academy of Sciences \cr
\email{martinkova@cs.cas.cz} \cr
}
\keyword{DDF}
