###########################################
#THIS FILE CONTAINS
#1) function for building o'neill and leibovici's entropy
#2) function for building the contagion index
#3) function for building parresol and edwards' entropy
###########################################


###########################################
#1) oneill and leibovici

#'O'Neill's and Leibovici's entropy.
#'
#'This function computes Leibovici's entropy according to a chosen distance \eqn{d}
#'(with O'Neill's entropy as a special case)
#'following Leibovici (2009), see also Altieri et al (2017). References can be found at \code{SpatEntropy}.
#'
#'This index is based on the transformed variable \eqn{Z} identifying couples of realizations
#'of the variable of interest. A distance of interest is fixed, which in the case of O'Neill's
#'entropy is the contiguity, i.e. sharing a border for lattice data; this corresponds to
#'creating an adjacency (contiguity) matrix as done by [adj_mat()]. Then, all couples
#'of realizations of the variable of interest identified by the adjacency matrix are counted
#'and their relative frequencies are used to compute the index with the traditional Shannon's
#'formula. Couples can be ordered or not (pairs), with ordered couples as the default option
#'being the authors' choice.
#'
#' @param data A data matrix or vector, can be numeric, factor, character, ...
#'   If the dataset is a point pattern, `data` is the mark vector.
#' @param adj.mat An adjacency matrix, upper- or lower-triangular.
#'                Provided by user or generated by [adj_mat()].
#' @param missing.cat Optional, a vector with the names of all categories that are
#'   absent in `data`.
#' @param ordered Logical, \code{T} if the entropy is computed using ordered couples
#' (see [couple_count()]), \code{F} if it is computed using pairs (see [pair_count()]).
#'
#' @return Leibovici's spatial entropy value (O'Neill's entropy when contiguity is considered)
#' as well as a summary table containing the couples (or pairs)
#' along with their absolute and relative frequencies.
#'
#' @aliases `oneill`
#'
#' @examples
#'##O'NEILL
#' dist.mat=euclid_dist(cbind(rep(1:5, each=5), rep(1:5,5)))
#' adj.mat=adj_mat(dist.mat, dd1=dist.mat[1,2]) #for the contiguity matrix
#' data=sample(1:3, 25, replace=TRUE)
#' oneill=leibovici(data, adj.mat)
#'
#'##LEIBOVICI
#' dist.mat=euclid_dist(cbind(rep(1:5, each=5), rep(1:5,5)))
#' adj.mat=adj_mat(dist.mat, dd1=4) #for the contiguity matrix
#' data=sample(1:3, 25, replace=TRUE)
#' leib=leibovici(data, adj.mat)
#'
#' @export

leibovici=function(data, adj.mat, missing.cat=NULL, ordered=TRUE)
{
  if(ordered==T) output=couple_count(data, adj.mat, missing.cat) else
    output=pair_count(data, adj.mat, missing.cat)
  prob=output$probabilities$proportion[output$probabilities$proportion>0]
  HH=-sum(prob*log(prob))
  return(list(freq.table=output$probabilities, entropy=HH))
}
###########################################

###########################################
#2) contagion

#'Li and Reynolds' relative contagion index.
#'
#'This function computes Li and Reynold's contagion index, following Li and Reynolds (1993),
#'starting from data or from the output of [leibovici()]. References can be found at \code{SpatEntropy}.
#'
#'This index is based on the transformed variable \eqn{Z} identifying couples of realizations
#'of the variable of interest. A distance of interest is fixed: the contagion index is
#'originally thought for areas sharing a border, as O'Neill's entropy. This corresponds to
#'creating a contiguity matrix as done by [adj_mat()]. Then, all couples
#'of realizations of the variable of interest identified by the adjacency matrix are counted
#'and their relative frequencies are used to compute the index, which is \eqn{1-NO} where \eqn{NO}
#'is the normalized O'Neill's entropy, i.e. O'Neill's entropy divided by its maximum \eqn{\log(I)},
#'\eqn{I} being the number of categories of the variable under study.
#'Couples can be ordered or not (pairs), with ordered couples as the default option
#'since it is the authors' choice. This function also allows to compute contagion for neighbourhood structures
#'different from contiguity, by suitably defining the adjacency matrix.
#'
#' @param oneill O'Neill's entropy as the output of [leibovici()].
#'               If this is provided, nothing else needs to be specified, except for `n.cat` if wished.
#' @param n.cat Optional, an integer denoting the number of categories of the study variable.
#' @param data A data matrix or vector, can be numeric, factor, character, ...
#'   If the dataset is a point pattern, `data` is the mark vector.
#' @param adj.mat The contiguity matrix, upper- or lower-triangular.
#'                Provided by user or generated by [adj_mat()].
#' @param missing.cat Optional, a vector with the names of all categories that are absent in `data`.
#' @param ordered Logical, \code{T} if the entropy is computed using ordered couples
#' (see [couple_count()]), \code{F} if it is computed using pairs (see [pair_count()]).
#'
#' @return Li and Reynolds' contagion index, as well as a summary table containing the couples (or pairs)
#' along with their absolute and relative frequencies.
#'
#' @examples
#'
#' dist.mat=euclid_dist(cbind(rep(1:5, each=5), rep(1:5,5)))
#' adj.mat=adj_mat(dist.mat, dd1=dist.mat[1,2]) #for the contiguity matrix
#' data=sample(1:3, 25, replace=TRUE)
#' oneill=leibovici(data, adj.mat)
#'
#' contag=contagion(oneill)
#'
#' contag=contagion(data=data, adj.mat=adj.mat)
#'
#' @export

contagion=function(oneill=NULL, n.cat=NULL, data=NULL, adj.mat=NULL, missing.cat=NULL, ordered=TRUE)
{
  if (!is.null(oneill)){
    if (is.null(n.cat)) n.cat=length(oneill$freq.table$proportion[oneill$freq.table$proportion>0])
    contagion=1-oneill$entropy/log(n.cat)
    output=list(freq.table=oneill$freq.table, contagion=contagion)
  }else{
    if(ordered==T) output=couple_count(data, adj.mat, missing.cat) else
      output=pair_count(data, adj.mat, missing.cat)
    prob=output$probabilities$proportion[output$probabilities$proportion>0]
    HH=-sum(prob*log(prob))
    contagion=1-HH/log(length(prob))
    output=list(freq.table=output$probabilities, contagion=contagion)
  }

  return(output)
}
###########################################

###########################################
#3) parresol

#'Parresol and Edwards' entropy.
#'
#'Compute Parresol and Edwards' entropy, following Parresol and Edwards (2014),
#'starting from data or from the output of [leibovici()]. References can be found at \code{SpatEntropy}.
#'
#'This index is based on the transformed variable \eqn{Z} identifying couples of realizations
#'of the variable of interest. A distance of interest is fixed: Parresol and Edwards' entropy is
#'originally thought for areas sharing a border, as O'Neill's entropy. This corresponds to
#'creating a contiguity matrix as done by [adj_mat()]. Then, all couples
#'of realizations of the variable of interest identified by the adjacency matrix are counted
#'and their relative frequencies are used to compute the index, which is the opposite of O'Neill's entropy.
#'Couples can be ordered or not (pairs), with ordered couples as the default option
#'since it is the authors' choice. This function also allows to compute Parresol and Edwards' entropy
#'for neighbourhood structures different from contiguity, by specifying a suitable adjacency matrix.
#'
#' @param oneill O'Neill's entropy as the output of [leibovici()].
#'               If this is provided, nothing else needs to be specified, except for `n.cat` if wished.
#' @param n.cat Optional, an integer denoting the number of categories of the study variable.
#' @param data A data matrix or vector, can be numeric, factor, character, ...
#'   If the dataset is a point pattern, `data` is the mark vector.
#' @param adj.mat The contiguity matrix, upper- or lower-triangular.
#'                Provided by user or generated by [adj_mat()].
#' @param missing.cat Optional, a vector with the names of all categories that are absent in `data`.
#' @param ordered Logical, \code{T} if the entropy is computed using ordered couples
#' (see [couple_count()]), \code{F} if it is computed using pairs (see [pair_count()]).
#'
#' @return Parresol and Edwards' spatial entropy value as well as a summary table  containing the couples
#' (or pairs) along with their absolute and relative frequencies.
#'
#' @examples
#'
#' dist.mat=euclid_dist(cbind(rep(1:5, each=5), rep(1:5,5)))
#' adj.mat=adj_mat(dist.mat, dd1=dist.mat[1,2]) #for the contiguity matrix
#' data=sample(1:3, 25, replace=TRUE)
#' oneill=leibovici(data, adj.mat)
#'
#' parr=parresol(oneill, n.cat=3)
#'
#' parr=parresol(data=data, adj.mat=adj.mat)
#'
#' @export

parresol=function(oneill=NULL, n.cat=NULL, data=NULL, adj.mat=NULL, missing.cat=NULL, ordered=TRUE)
{
  if (!is.null(oneill)){
    parresol=-oneill$entropy
    output=list(freq.table=oneill$freq.table, parresol=parresol)
  }else{
    if(ordered==T) output=couple_count(data, adj.mat, missing.cat) else
      output=pair_count(data, adj.mat, missing.cat)
    prob=output$probabilities$proportion[output$probabilities$proportion>0]
    HH=-sum(prob*log(prob))
    parresol=-HH
    output=list(freq.table=output$probabilities, parresol=parresol)
  }

  return(output)
}
###########################################
